import compiler, traceback, os.path

###########################################################################
## External API

def compile(filename):
    """
    Compile a Python source file to IMCC instructions
    """
    main=os.path.basename(filename).split('.')[0]

    push()
    output(".sub %s::__main__ @MAIN" % main)
    output("\t.param pmc sys::argv")
    execute(compiler.parseFile(filename))
    output(".end")
    pop()
   
###########################################################################
## Internal API

blocks=[]
def push():
    """
    Suspend current block, start a new block
    """
    blocks.append(block())

def pop():
    """
    Complete current block, resume previous block
    """
    print "\n".join(blocks.pop().instructions)
    if blocks: print

def push_loop(info):
    """
    Push information on a loop stack
    """
    blocks[-1].loop.append(info)

def peek_loop():
    """
    Pop information from the loop stack
    """
    return blocks[-1].loop[-1]

def pop_loop():
    """
    Pop information from the loop stack
    """
    return blocks[-1].loop.pop()

def lookup(symbol, init=True):
    """
    Lookup a symbol in the current scope
    """
    return blocks[-1].lookup(symbol, init)

def register(symbol,name):
    """
    Register a symbol in the current scope
    """
    blocks[-1].symbols[name]=symbol

def param(name):
    """
    Define a parameter to the local block
    """
    output("\t.param pmc " + name)
    register(name,name)

def label():
    """
    Return a unique label
    """
    blocks[-1].counter += 1
    return "l" + str(blocks[-1].counter)

def temp():
    """
    Return a temporary variable
    """
    blocks[-1].temp += 1
    return "$P" + str(blocks[-1].temp)

def execute(node):
    """
    execute (emit instructions for) a given compiler.ast node
    """
    map[node.__class__](node)

def evaluate(node):
    """
    evaluate (return symbol/constant/register for) a given compiler.ast node
    """
    return map[node.__class__](node)

def output(instruction):
    """
    output a single IMCC instruction
    """

    # pad right
    col=len(instruction)
    if instruction[0]=='\t': col+=7
    while col<24:
        instruction+='\t'
        col+=8

    # determine caller
    (file,line,func)=traceback.extract_stack()[-2][:3]
    file=os.path.basename(file).split('.')[0]

    # append to current block
    blocks[-1].append("%s\t# %s.%s:%d" % (instruction, file, func, line))

def fail(message):
    """
    report an error
    """
    print "***", message

###########################################################################

class block:
  def __init__(self):
    self.instructions=[]
    self.symbols={}
    self.counter=0
    self.loop=[]
    self.temp=0

  def lookup(self, symbol, init):
    if symbol not in self.symbols:
      if symbol in ['num']:
        self.temp+=1
        reg="$P"+str(self.temp)
        self.symbols[symbol]=reg
        # TODO output("\tstore_lex -1, \"%s\", %s" % (symbol, reg))
      else:
        output("\t.local pmc " + symbol)
        self.symbols[symbol]=symbol
      if init: output("\t%s = new PerlInt" % self.symbols[symbol])
    return self.symbols[symbol]

  def append(self, instruction):
    self.instructions.append(instruction)

#
# create a map between compiler.ast classes and pyrate.ast process functions
# for efficient dispatch
#
map={}
for node in compiler.ast.klasses.keys():
    if node[0].isupper() and node[1].islower():
        try:
            process=getattr(getattr(__import__("ast."+node),node),"process")
            map[compiler.ast.klasses[node]]=process
        except:
            print node
